<?php

// Namespace to Models folder
namespace App\Models;

// Call Model Namespace
use CodeIgniter\Model;


// begin::BaseModel class
class BaseModel extends Model
{

    /* Model configuration */
    protected $returnType     = 'array';
    protected $useSoftDeletes = true;

    /* Logs columns */
    protected $useTimestamps = false;
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    

    /**
     * LIST ALL DATA AVAILABLE
     * OR A SPECIFIED ROW DETAILS
     * 
     * @param   int    $rowID
     * @param   array  $options
     * 
     * @return  array  $response
     */
    protected function _list(int $rowID = 0, array $options = []):?array
    {

        // Set the default data to return
        $data = [];

        // Set the fields list to select
        $this->select($this->selectedFields);


        /**
         * JOINTS
         * 
         * Set all the joins for the current query
         */
        $this->_setJoints();


        // Get all the received options
        $paging = (! empty($options) && ! empty($options['paging'])) ? $options['paging'] : [];
        $page = (! empty($paging) && ! empty($paging['page']) && intval($paging['page']) > 0) ? intval($paging['page']) : 1;
        $length = (! empty($paging) && ! empty($paging['length']) && intval($paging['length']) > 0) ? intval($paging['length']) : 10;
        $sorting = (! empty($options) && ! empty($options['sorting'])) ? $options['sorting'] : [];
        $field = (! empty($sorting) && ! empty($sorting['field'])) ? mb_strtolower($sorting['field']) : 'created_at';
        $sort = (! empty($sorting) && ! empty($sorting['sort']) && mb_strtoupper($sorting['sort']) === 'ASC') ? 'ASC' : 'DESC';
        $status = (! empty($options['status']) || $options['status'] === '0') ? intval($options['status']) : 'all';

        // Check wether an id has been specified or not
        if($rowID === 0) {

            // Adjust the query if the paging option is specified
            if(! empty($page)) {

                // Set the offset value
                $offset = ($page - 1) * $length;

                // Add paging option
                $this->limit($length, $offset);

            } // End if

            // Adjust the query if the sorting option is specified
            if(! empty($field)
                && array_key_exists($field, $this->sortingFields)
            ) {

                // Add sorting option
                $this->orderBy($this->sortingFields[$field], $sort);

            } // End if

            // Adjust the query according to the received status
            if($status !== 'all') {

                // Add condition
                $this->where([
                    $this->table . '.active'    => intval($status)
                ]);

            } // End if

            // Returns all the rows availables
            $data = $this->where([
                            $this->table . '.deleted'       => 0,
                            $this->table . '.deleted_by'    => null,
                        ])
                        ->get()
                        ->getResultArray();

        } // End if
        else {

            // Return the row details
            $data = $this->asArray()
                        ->where([
                            $this->table . '.' . $this->primaryKey   => $rowID,
                        ])
                        ->first(); // Return the first item
        } // End if

        // Set the response to return
        $response = [
            'data'      => $data,
            'page'      => $page,
            'length'    => $length,
            'field'     => $field,
            'sort'      => $sort,
        ];

        // Return the response
        return $response;

    } // end::list

    

    /**
     * COUNT ALL THE ROWS
     * AVAILABLE
     * 
     * @param   string  $keyword
     * @param   bool    $allFields
     * 
     * @return  int     $total
     */
    protected function _countTotal(string $keyword = '', bool $allFields = true):?int
    {

        // Set the fields list to select
        $this->select($this->table . '.' . $this->primaryKey);


        /**
         * JOINTS
         * 
         * Set all the joins for the current query
         */
        $this->_setJoints();


        // Process only if the keyword is not empty
        // Search through all the availables fields if required
        if($allFields && ! empty($keyword))
        {

            // Get the search array
            $searchArray = $this->_searchArray($keyword);

            // Return the rows that match any column
            return $this->orLike($searchArray)
                        ->countAllResults();

        } // End if

        // Return the rows that match even if the keyword is empty
        return $this->like($this->searchFields[0], $keyword)
                    ->countAllResults();

    } // end::_countTotal

    

    /**
     * LIST ALL THE ROWS
     * THAT MATCHS KEYWORD
     * 
     * @param   string  $keyword
     * @param   bool    $allFields
     * @param   array   $options
     * 
     * @return  array   $response
     */
    protected function _search(string $keyword = '', bool $allFields = false, array $options = []):?array
    {

        // Set the default data to return
        $data = [];

        // Set the fields list to select
        $this->select($this->selectedFields);


        /**
         * JOINTS
         * 
         * Set all the joins for the current query
         */
        $this->_setJoints();


        // Get all the received options
        $paging = (! empty($options) && ! empty($options['paging'])) ? $options['paging'] : [];
        $page = (! empty($paging) && ! empty($paging['page']) && intval($paging['page']) > 0) ? intval($paging['page']) : 1;
        $length = (! empty($paging) && ! empty($paging['length']) && intval($paging['length']) > 0) ? intval($paging['length']) : 10;
        $sorting = (! empty($options) && ! empty($options['sorting'])) ? $options['sorting'] : [];
        $field = (! empty($sorting) && ! empty($sorting['field'])) ? mb_strtolower($sorting['field']) : 'created_at';
        $sort = (! empty($sorting) && ! empty($sorting['sort']) && mb_strtoupper($sorting['sort']) === 'ASC') ? 'ASC' : 'DESC';
        $status = (! empty($options['status']) || $options['status'] === '0') ? intval($options['status']) : 'all';

        // Adjust the query if the paging option is specified
        if(! empty($page))
        {

            // Set the offset value
            $offset = ($page - 1) * $length;

            // Add paging option
            $this->limit($length, $offset);

        } // End if

        // Adjust the query if the sorting option is specified
        if(! empty($field)
            && array_key_exists($field, $this->sortingFields)
        )
        {

            // Add sorting option
            $this->orderBy($this->sortingFields[$field], $sort);

        } // End if

        // Return only non-deleted rows
        $this->where([
            $this->table . '.deleted'       => 0,
            $this->table . '.deleted_by'    => null,
        ]);

        // Adjust the query according to the received status
        if($status !== 'all') {

            // Add condition
            $this->where([
                $this->table . '.active'    => intval($status)
            ]);

        } // End if

        // Search through all the availables fields if required
        if($allFields && ! empty($keyword))
        {

            // Get the search array
            $searchArray = $this->_searchArray($keyword);

            // Return the rows that match any column
            $data = $this->groupStart()
                            ->orLike($searchArray)
                        ->groupEnd()
                        ->get()
                        ->getResultArray();

        } // End if
        else {

            // Process only if the keyword is not empty
            $data = $this->like($this->searchFields[0], $keyword)
                        ->get()
                        ->getResultArray();

        } // End else

        // Set the response to return
        $response = [
            'data'      => $data,
            'page'      => $page,
            'length'    => $length,
            'field'     => $field,
            'sort'      => $sort,
        ];

        // Return the response
        return $response;

    } // end::_search

    

    /**
     * SET ROW
     * 
     * @param   int    $rowID
     * @param   array  $data
     * 
     * @return  array  $response
     */
    protected function _set(int $rowID = 0, array $data = []):?array
    {

        // Return the function response
        $type = 'danger';
        $message = lang('General.response.badRequest') . ' .';

        // Get all the required data
        $code = (! empty($data) && ! empty($data['code'])) ? $data['code'] : '';
        $groupID = (! empty($data) && ! empty($data['group']) && intval($data['group']) > 0) ? intval($data['group']) : 0;

        // Process only if the required data are availables
        if(! empty($code) && ! empty($groupID)) {

            // Get the other data
            $rowID = (! empty($rowID) && intval($rowID) > 0) ? intval($rowID) : 0;
            $tags = (! empty($data) && ! empty($data['tags'])) ? $data['tags'] : '';
            $description = (! empty($data) && ! empty($data['description'])) ? $data['description'] : '';

            // Check if this is not duplicate data
            if(! $this->isDuplicate([
                $this->table . '.' . $this->primaryKey . ' !='  => $rowID,
                $this->table . '.' . 'code'                     => $code,
                $this->table . '.' . 'deleted'                  => 0,
                $this->table . '.' . 'deleted_at'               => null,
                $this->table . '.' . 'deleted_by'               => null
            ]))
            {

                // Check wether we're facing an insertion or an update
                if(! empty($rowID))
                { // Update

                    // Try to get the row that correspond to the specified id
                    $existingRow = $this->list($rowID);

                    // Check wether the row exist or not
                    if(! empty($existingRow))
                    {

                        // Add the new destination
                        $updated = $this->update($rowID, [
                            'code'          => $code,
                            'group_id'      => $groupID,
                            'tags'          => $tags,
                            'description'   => $description,
                        ]);

                        // Check if the row has been successfully created
                        if($updated)
                        {

                            // Set the response details
                            $type = 'success';
                            $message = lang('General.response.updated');

                        } // End if
                        else
                        {

                            // Set the response details
                            $type = 'warning';
                            $message = lang('General.response.server');

                        } // End else

                    } // End if
                    else
                    {

                        // Set the response details
                        $type = 'danger';
                        $message = lang('General.response.notFound');

                    } // End else

                } // End if
                else
                { // Creation

                    // Add the new row
                    $created = $this->save([
                        'group_id'      => $groupID,
                        'code'          => $code,
                        'tags'          => $tags,
                        'description'   => $description,
                        'active'        => 1,
                    ]);

                    // Check if the row has been successfully created
                    if($created)
                    {

                        // Set the response details
                        $type = 'success';
                        $message = lang('General.response.created');

                    } // End if
                    else
                    {

                        // Set the response details
                        $type = 'warning';
                        $message = lang('General.response.server');

                    } // End else

                } // End else

            } // End if
            else
            {

                // Set the response details
                $type = 'danger';
                $message = lang('General.response.duplicated');

            } // End else

        } // End if

        // Set the response
        $response = [
            'type'      => $type,
            'message'   => $message,
        ];

        // Return the response
        return $response;

    } // end::_set

    

    /**
     * UDPATE ROW STATUS
     * ENABLE / DISABLE
     * 
     * @param   int    $rowID
     * @param   int    $status
     * 
     * @return  array  $response
     */
    protected function _status(int $rowID = 0, int $status = 0):?array
    {

        // Set the default response data
        $code = 'danger';
        $message = lang('General.response.notFound');

        // Get all the sent informations
        $rowID = (! empty($rowID) && intval($rowID) > 0) ? intval($rowID) : 0;
        $status = (! empty($status) && intval($status) > 0) ? intval($status) : 0;

        // Process only if the required data are available
        if(! empty($rowID)) {

            /**
             * CHECK IF THE SPECIFIED ROW DOES EXIST
             */

            // Get the row details
            $rowDetails = $this->_getDataBy([
                'id'            => $rowID,
                'deleted'       => INJECTION,
                'deleted_at'    => null,
            ]);

            // Process only if the row details aray is not empty
            if(! empty($rowDetails)) {

                // Try to update the specified row
                $rowUpdated = $this->update($rowID, [
                    'active'    => $status // Field must be the same as the allowed defined ones
                ]);

                // Process only if the row has been successfully updated
                if($rowUpdated) {

                    // Set the response details
                    $code = 'success';
                    $message = lang('General.response.statusUpdated');

                } // End if
                else {

                    // Set the response details
                    $code = 'danger';
                    $message = lang('General.response.server');


                } // End else

            } // End if
            else {

                // Set the response details
                $code = 'warning';
                $message = lang('General.response.notFound');

            } // End else

        } // End if

        // Set the response
        $response = [
            'code'      => $code,
            'message'   => $message
        ];

        // Retunr the response
        return $response;

    } // end::_status

    

    /**
     * DELETE A ROW
     * 
     * @param   int    $rowID
     * 
     * @return  array  $response
     */
    protected function _delete(int $rowID = 0):?array
    {

        // Return the function response
        $code = 'error';
        $message = lang('General.response.badRequest');

        // Process only if the required row's ID is not empty
        if(! empty($rowID) && intval($rowID) > 0) {

            // Try to get the row that correspond to the specified row's ID
            $existingRow = $this->_getDataBy([
                'id'            => $rowID,
                'deleted'       => INJECTION,
                'deleted_at'    => null,
            ]);

            // Check wether the row exist or not
            if(! empty($existingRow))
            {

                // Add the new route
                $deleted = $this->delete($rowID);

                // Check if the row has been successfully created
                if($deleted)
                {

                    // Set the message to return
                    $code = 'success';
                    $message = lang('General.response.deleted');

                    // Update the delete informations details
                    $this->update($rowID, [
                        'deleted'       => 1,
                    ]);

                } // End if
                else
                {

                    // Set the response details
                    $code = 'warning';
                    $message = lang('General.response.server');

                } // End else

            } // End if
            else
            {

                // Set the response details
                $code = 'error';
                $message = lang('General.response.notFound');

            } // End else

        } // End if

        // Set the response
        $response = [
            'code'      => $code,
            'message'   => $message,
        ];

        // Return the function response
        return $response;

    } // end::_delete

    

    /**
     * CHECK WETHER A ROW IS A DUPLICATE OR NOT
     * ACCORDING TO A SPECIFIC QUERY
     * 
     * @param   array  $where   => Where queries using AND
     * @param   array  $orWhere => Where queries using OR
     * 
     * @return  bool   $isDuplicate
     */
    protected function _isDuplicate(array $where = [], array $orWhere = []):?bool
    {

        // Set the fields list to select
        $this->select($this->table . '.' . $this->primaryKey);


        /**
         * JOINTS
         * 
         * Set all the joins for the current query
         */
        $this->_setJoints();


        // Set the default answer to return
        $isDuplicate = true;

        // Process only if there is a orWhere available
        if(! empty($orWhere))
        {
            
            $this->groupStart()
                    ->orWhere($orWhere)
                ->groupEnd();

        } // End if

        // Try to get the row
        $row = $this->where($where)
                    ->get()
                    ->getResultArray();

        // Check the row
        if( empty($row))
        {

            // Set the row as not duplicate
            $isDuplicate = false;

        } // End if

        // Return the answer
        return $isDuplicate;

    } // end::_isDuplicate

    

    /**
     * GET ANY ROW THAT MATCHES THE SENT 
     * FILTERS ACCORDING THE LOGIC VALUE
     * 
     * @param   array  $filters
     * @param   bool   $logicOR
     * 
     * @return  array  $data
     */
    protected function _getDataBy(array $filters, bool $logicOR = false):?array
    {

        // Set the default data value
        $data = [];
        
        // Get the relatives conditions
        $conditions = $this->_conditionsFromFilters($filters);

        // Test
        // ...
        // var_dump($conditions);


        /**
         * JOINTS
         * 
         * Set all the joins for the current query
         */
        $this->_setJoints();

        // Process only if there is a condition specified
        if(! empty($conditions)) {

            // Get the data specified
            $data = $this->asArray()
                        ->select($this->selectedFields)
                        ->where($conditions)
                        ->first();

        } // End if

        return $data;

    } // end::_getDataBy

    

    /**
     * TURN FILTERS ARRAY
     * INTO CONDITIONS ARRAY
     * 
     * @param   array  $filters
     * 
     * @return  array  $conditions
     */
    protected function _conditionsFromFilters(array $filters = []):?array
    {

        // Set the default condition value
        $conditions = [];

        // Process if there is any filter specified and sent
        if(! empty($filters) && ! empty($this->filtersFields)) {

            // Loop through all the filters then set the condition that fits
            foreach($filters as $key => $value) {

                // Check if the key matches any filters specified
                if(array_key_exists($key, $this->filtersFields)) {

                    // Set the condition
                    $conditions[$this->filtersFields[$key]] = $value;

                } // End if

            } // End loop

        } // End if

        // Return the response
        return $conditions;

    } // end::_conditionsFromFilters

    

    /**
     * SET ARRAY OF COLUMN WHERE
     * TO LOOK FOR THE KEYWORD
     * 
     * @param   string  $keyword
     * 
     * @return  array   $data
     */
    protected function _searchArray(string $keyword = ''):?array
    {

        // Set the default data to return
        $data = [];

        // Loop through all the fields then add the keyword as value
        foreach($this->searchFields as $searchField)
        {
            
            // Set a new array item
            $data["$searchField"] = $keyword;

        } // End loop

        // Return data as function response
        return $data;

    } // end::_searchArray

    

    /**
     * SET TABLE JOINTS
     * 
     * @return  void
     */
    protected function _setJoints()
    {

        // Process only if there is any joint available
        if(! empty($this->joints)) {

            // Loop through all the joints
            for($i=0; $i < count($this->joints); $i++) {

                // Process only if the required data are available
                if(! empty($this->joints[$i]['table']) && $this->joints[$i]['condition']) {

                    // Set the JOIN condition
                    $this->join($this->joints[$i]['table'], $this->joints[$i]['condition'], $this->joints[$i]['type']);

                } // End if

            } // End loop

        } // End if

    } // end::_setJoints

    

    /**
     * GENERATE RANDOM SALT KEY
     * 
     * @return  string  $randomSalt
     */
    protected function _randomSalt():?string
    {

        // Return the key
        return md5(mt_rand());

    } // end::_randomSalt

    

    /**
     * HASH A PASSWORD USING
     * RANDOM SALT KEY
     * 
     * @param   string  $password
     * @param   string  $randomSalt
     * 
     * @return  string  $hashedPassword
     */
    protected function _hashPassword(string $password, string $randomSalt):?string
    {

        /**
         * bcrypt is the preferred hashing for passwords, but
         * is only available for PHP 5.3+. Even in a PHP 5.3+ 
         * environment, we have the option to use PBKDF2; just 
         * set the PHP52_COMPATIBLE_PASSWORDS constant located 
         * in config/constants.php to 1.
         */

        // Load the Encryption Library
        $encrypter = \Config\Services::encrypter();

        // Return the encrypted password
        return crypt($password . $encrypter->key, '$2a$09$' . $randomSalt . '$');

    } // end::_hashPassword


} // end::BaseModel class