<?php

// Namespace to Models folder
namespace App\Models;

// Call BaseModel Namespace
use App\Models\BaseModel;

// Load required models
use App\Models\UsersIdentifiersModel; // users identifiers
use App\Models\RolesPermissionsModel; // roles permissions


// begin::UsersModel class
class UsersModel extends BaseModel
{

    // Model configuration
    protected $table = 'user';
    protected $primaryKey = 'id';
    
    // Allowed fields for insertion
    protected $allowedFields = ['id', 'role_id', 'first_name', 'last_name', 'phone', 'email', 'pwd', 'random_salt', 'active', 'deleted', 'created_by', 'deleted_by', 'created_at', 'deleted_at', 'updated_at'];
    
    // Fields available on SELECT query
    protected $selectedFields = 'user.id, user.first_name, user.last_name, user.phone, user.email, user.random_salt, user.active, user.deleted, user.created_by, user.deleted_by, user.created_at, user.deleted_at, user.updated_at, role.id as role_id, role.name as role_name';

    // Fields used on filters queries
    // Used by _getDataBy() function
    protected $filtersFields = [
        'id'        => 'user.id',
        'pwd'       => 'user.pwd',
        'active'    => 'user.active',
        'deleted'   => 'user.deleted',
    ];

    // Fields used for SEARCH query
    protected $searchFields = ['user.first_name', 'user.last_name', 'user.phone', 'user.email', 'user.created_at'];

    // Fields available for sorting
    protected $sortingFields = [
        'first_name'    => 'user.first_name',
        'last_name'     => 'user.last_name',
        'phone'         => 'user.phone',
        'email'         => 'user.email',
        'active'        => 'user.active',
        'created_at'    => 'user.created_at',
    ];

    // Set the joins array
    protected $joints = [
        [
            'table'     => 'role',
            'condition' => 'role.id = user.role_id',
            'type'      => 'left',
        ],
    ];



    /**
     * GET DATA LIST
     * 
     * @param   int    $rowID
     * @param   array  $options
     * 
     * @return  array  $response
     */
    public function list(int $rowID = 0, array $options = [])
    {

        // Get the row ID
        $rowID = (! empty($rowID) && intval($rowID) > 0) ? intval($rowID) : 0;

        // Process only if the row ID has been specified
        if($rowID === 0) {

            // Get all the received data
            $paging = (! empty($options['paging'])) ? $options['paging'] : [];
            $sorting = (! empty($options['sorting'])) ? $options['sorting'] : [];
            $search = (! empty($options['search'])) ? trim($options['search']) : '';
            $status = (! empty($options['status']) || $options['status'] === '0') ? intval($options['status']) : 'all';

            // Check if any keyword has been sent for search
            if(! empty($search)) {

                // Load the requested data
                $responseData = $this->_search($search, true, [
                    'status'    => $status,
                    'paging'    => $paging,
                    'sorting'   => $sorting,
                ]);

            } // End if
            else {

                // Load the requested data
                $responseData = $this->_list($rowID, [
                    'status'    => $status,
                    'paging'    => $paging,
                    'sorting'   => $sorting,
                ]);

            } // End else

            // Get the total rows availables
            $total = $this->_countTotal($search);

            // Get the total pages availables
            $pages = ceil($total / $responseData['length']);

            $page = ($responseData['page'] > $pages) ? $pages : $responseData['page'];

            // Set the response to return
            $response = [
                'meta'  => [
                    'page'      => $page,
                    'pages'     => $pages,
                    'perpage'   => $responseData['length'],
                    'total'     => $total,
                    'sort'      => $responseData['sort'],
                    'field'     => $responseData['field'],
                ],
                'data'  => $this->_formatRows($responseData['data'], true),
            ];

        } // End if
        else {

            // Load the requested data
            $responseData = $this->_getDataBy([
                'id'    => $rowID
            ]);

            // Set the response to return
            $response = [
                'meta'  => [],
                'data'  => $this->_formatRowData($responseData),
            ];

        } // End else

        // Return the response
        return $response;

    } // end::list

    

    /**
     * SET ROW
     * 
     * @param   int    $rowID
     * @param   array  $data
     * 
     * @return  array  $response
     */
    public function setRow(int $rowID = 0, array $data = []):?array
    {

        // Return the function response
        $code = 'danger';
        $message = lang('General.response.badRequest') . ' .';

        // Get all the required data
        $lastName = (! empty($data) && ! empty($data['lastName'])) ? $data['lastName'] : '';
        $roleID = (! empty($data) && ! empty($data['role']) && intval($data['role']) > 0) ? intval($data['role']) : 0;
        $email = (! empty($data) && ! empty($data['email'])) ? $data['email'] : '';
        $phone = (! empty($data) && ! empty($data['phone'])) ? $data['phone'] : '';

        // Process only if the required data are availables
        if(! empty($lastName) && ! empty($roleID) && ! empty($email) && ! empty($phone)) {

            // Get the other data
            $rowID = (! empty($rowID) && intval($rowID) > 0) ? intval($rowID) : 0;
            $firstName = (! empty($data) && ! empty($data['firstName'])) ? $data['firstName'] : '';
            $password = (! empty($data) && ! empty($data['password'])) ? $data['password'] : '';

            // Check if this is not duplicate data
            if(! $this->_isDuplicate([
                    $this->table . '.' . $this->primaryKey . ' !='  => $rowID,
                    $this->table . '.' . 'deleted'                  => INJECTION,
                    $this->table . '.' . 'deleted_at'               => null,
                    $this->table . '.' . 'deleted_by'               => null
                ], [
                    $this->table . '.email'  => $email,
                    $this->table . '.phone'  => $phone,
                ])
            ) {

                // Check wether we're facing an insertion or an update
                if(! empty($rowID))
                { // Update

                    // Try to get the row that correspond to the specified id
                    $existingRow = $this->list($rowID);

                    // Check wether the row exist or not
                    if(! empty($existingRow))
                    {

                        // Add the new destination
                        $updated = $this->update($rowID, [
                            'role_id'       => $roleID,
                            'first_name'    => $firstName,
                            'last_name'     => $lastName,
                            'email'         => $email,
                            'phone'         => $phone,
                        ]);

                        // Check if the row has been successfully created
                        if($updated)
                        {

                            // Set the response details
                            $code = 'success';
                            $message = lang('General.response.updated');

                        } // End if
                        else
                        {

                            // Set the response details
                            $code = 'warning';
                            $message = lang('General.response.server');

                        } // End else

                    } // End if
                    else
                    {

                        // Set the response details
                        $code = 'danger';
                        $message = lang('General.response.notFound');

                    } // End else

                } // End if
                else
                { // Creation

                    // Generate password for the user
                    // $password = $this->_generatePassword();
                    $randomSalt = $this->_randomSalt();

                    // Add the new row
                    $rowID = $this->insert([
                        'role_id'       => $roleID,
                        'first_name'    => $firstName,
                        'last_name'     => $lastName,
                        'email'         => $email,
                        'phone'         => $phone,
                        'random_salt'   => $randomSalt,
                        'pwd'           => $this->_hashPassword($password, $randomSalt),
                        'active'        => ACTIVE,
                    ]);

                    // Check if the row has been successfully created
                    if(! empty($rowID))
                    {

                        // Load the required models
                        $usersIdentifiersModel = new UsersIdentifiersModel(); // Users Identifiers Model

                        // Add identifiers for user connection
                        $usersIdentifiersModel->addIdentifiers($rowID, [$email, $phone]);

                        // Add identifiers for user connection
                        // Email
                        // $usersIdentifiersModel->save([
                        //     'user_id'   => $rowID,
                        //     'name'      => $email,
                        //     'active'    => ACTIVE,
                        // ]);
                        // // Phone
                        // $usersIdentifiersModel->save([
                        //     'user_id'   => $rowID,
                        //     'name'      => $phone,
                        //     'active'    => ACTIVE,
                        // ]);

                        // Set the response details
                        $code = 'success';
                        $message = lang('General.response.created');

                    } // End if
                    else
                    {

                        // Set the response details
                        $code = 'warning';
                        $message = lang('General.response.server');

                    } // End else

                } // End else

            } // End if
            else
            {

                // Set the response details
                $code = 'danger';
                $message = lang('General.response.duplicated');

            } // End else

        } // End if

        // Set the response
        $response = [
            'code'      => $code,
            'message'   => ucfirst($message),
        ];

        // Return the response
        return $response;

    } // end::set



    /**
     * UDPATE ROW STATUS
     * ENABLE / DISABLE
     * 
     * @param   int  $rowID
     * @param   int  $status
     * 
     * @return  string $response
     */
    public function setStatus(int $rowID = 0, int $status = 0)
    {

        // Update the row status
        $response = $this->_status($rowID, $status);

        // Return the response
        return $response;

    } // end::setStatus



    /**
     * DELETE ROW
     * 
     * @param   int  $rowID
     * 
     * @return  string $response
     */
    public function deleteRow(int $rowID = 0)
    {

        // Update the row status
        $response = $this->_delete($rowID);

        // Return the response
        return $response;

    } // end::deleteRow



    /**
     * AUTHENTICATE A USER
     * USING CREDENTIALS
     * 
     * @param   array  $data
     * 
     * @return  array  $response
     */
    public function authentication(array $data = []):?array
    {

        // Set the default response to return
        $code = 'warning';
        $message = lang('SignIn.response.badCredentials');
        $responseData = '';

        // Get the received data
        $login = (! empty($data['login'])) ? trim($data['login']) : '';
        $password = (! empty($data['password'])) ? trim($data['password']) : '';

        // Process only if the required data are available
        if(! empty($login) && ! empty($password)) {

            // Load the required models
            $usersIdentifiersModel = new UsersIdentifiersModel(); // Users Identifiers Model

            // Check if the received login macthes any account from database
            $userAccount = $usersIdentifiersModel->isUserLogin($login);

            // Process only if the user account data are not empty
            if(! empty($userAccount)) {

                // Get the password informations
                $randomSalt = (! empty($userAccount['random_salt'])) ? trim($userAccount['random_salt']) : '';

                // Hash the received password
                $hashedPassword = $this->_hashPassword($password, $randomSalt);

                // Get any account informations using the received credentials
                $responseData = $this->_getDataBy([
                    'pwd'       => $hashedPassword,
                    'active'    => ACTIVE,
                    'deleted'   => INJECTION,
                ]);

                // Test
                // ...
                // var_dump('user account');
                // var_dump($userAccount);
                // var_dump($randomSalt);
                // var_dump($this->_hashPassword('admin', $randomSalt));

                // Process only if the response data is not empty
                if(! empty($responseData)) {

                    // Load the required models
                    $rolesPermissionsModel = new RolesPermissionsModel(); // Roles Permissions Model

                    // Load the role permissions
                    $responseData['permissions'] = $rolesPermissionsModel->_rolePermissions($responseData['role_id']);
                    // $responseData['permissions'] = $rolesPermissionsModel->_getDataBy([
                    //     'role_id'               => $responseData['role_id'],
                    //     'role_active'           => ACTIVE,
                    //     'role_deleted'          => INJECTION,
                    //     'permission_active'     => ACTIVE,
                    //     'permission_deleted'    => INJECTION,
                    // ]);

                    // Set the response details
                    $code = 'success';
                    $message = lang('SignIn.response.signedIn');

                } // End if

            } // End if

        } // End if

        // Set the response
        $response = [
            'code'      => $code,
            'message'   => ucfirst($message),
            'data'      => $responseData,
        ];

        // Return the response
        return $response;

    } // end::authentication

    

    /**
     * FORMAT USER PROFILE
     * 
     * @param   array  $data
     * 
     * @return  array  $formatedData
     */
    public function _formatProfileData(array $data = []):?array
    {

        // Set the default formatted data to return
        $formattedData = [];

        // Check whether the data to format is empty or not
        if(! empty($data))
        {

            // Process only if the sent data is an object
            if(is_object($data))
            {

                // Cast to array type
                $data = (array)$data;

            } // End if

            // Get all the formatted data
            $id = (! empty($data['id']) && intval($data['id']) > 0) ? intval($data['id']) : 0;
            $firstName = (! empty($data['first_name'])) ? trim($data['first_name']) : '';
            $lastName = (! empty($data['last_name'])) ? trim($data['last_name']) : '';
            $roleName = (! empty($data['role_name'])) ? trim($data['role_name']) : '';
            $permissionsArray = (! empty($data['permissions'])) ? $data['permissions'] : [];
            $permissions = [];
            $phone = (! empty($data['phone'])) ? trim($data['phone']) : '';
            $email = (! empty($data['email'])) ? trim($data['email']) : '';
            $photo = (! empty($data['photo'])) ? trim($data['photo']) : base_url() . '/assets/global/media/users/blank.png';
            $active = (! empty($data['active']) && intval($data['active']) > 0) ? intval($data['active']) : 0;


            // Test
            // ...
            // var_dump($data);
            // var_dump($permissionsArray);

            // Loop through all the permissions items then uppercase all
            foreach($permissionsArray as $permission)
            {

                // Test
                // ...
                // var_dump($permission);
                // var_dump($permission['permission_code']);
                // exit;

                // 
                $permissions[] = mb_strtoupper($permission['permission_code']);

            } // End loop

            // Set the formatted data
            $formattedData = [
                'id'            => $id,
                'firstName'     => $firstName,
                'lastName'      => $lastName,
                'phone'         => $phone,
                'email'         => $email,
                'photo'         => $photo,
                'role'          => $roleName,
                'permissions'   => $permissions,
                'active'        => $active
            ];

        } // End if

        return $formattedData;

    } // End private function

    

    /**
     * UPDATE USER PROFILE
     * 
     * @param   array  $data
     * 
     * @return  array  $response
     */
    public function _setProfile(array $data = []):?array
    {

        // Return the function response
        $message = ucfirst(lang('General.response.badRequest'));
        $code = 'warning';

        // Set the default data to return
        $responseData = []; // data

        // Get all the received data
        $rowID = (! empty($data['rowID']) && intval($data['rowID'])) ? intval($data['rowID']) : 0;
        $firstName = (! empty($data['firstName'])) ? $data['firstName'] : '';
        $lastName = (! empty($data['lastName'])) ? $data['lastName'] : '';
        $email = (! empty($data['email'])) ? $data['email'] : '';
        $phone = (! empty($data['phone'])) ? $data['phone'] : '';

        // Checker whether an ID has been provided or not
        if(! empty($rowID)
            && ! empty($firstName)
            && ! empty($lastName)
            && ! empty($email)
            && ! empty($phone)
        ) {

            // Get the profile to update
            $existingRow = $this->_list($rowID);

            // Check whether the profile exist or not
            if(! empty($existingRow)) {

                // Set the data array to update
                $toUpdateData = [
                    'first_name'    => $firstName,
                    'last_name'     => $lastName,
                    'email'         => $email,
                    'phone'         => $phone,
                ];

                // Update the profile
                $updated = $this->update($rowID, $toUpdateData);

                // Process if update succeed
                if($updated) {

                    // Set the response data
                    $responseData = [
                        'firstName' => $firstName,
                        'lastName'  => $lastName,
                        'email'     => $email,
                        'phone'     => $phone,
                    ];

                    // Set the informations to return
                    $code = 'success';
                    $message = ucfirst(lang('Profile.response.updated'));

                } // End if
                else {

                    // Set the informations to return
                    $code = 'warning';
                    $message = ucfirst(lang('General.response.server'));

                } // End else

            } // End if
            else {

                // Set the informations to return
                $code = 'warning';
                $message = ucfirst(lang('Profile.response.notFound'));

            } // End else

        } // End if

        // Set the current function response
        $response = [
            'code'      => $code,
            'message'   => $message,
            'data'      => $responseData,
        ];

        // Return the response
        return $response;

    } // End function

    

    /**
     * UPDATE USER PASSWORD
     * 
     * @param   array  $data
     * 
     * @return  array  $response
     */
    public function _setPassword(array $data = []):?array
    {

        // Return the function response
        $message = ucfirst(lang('General.response.badRequest'));
        $code = 'warning';

        // Set the default data to return
        $responseData = []; // data

        // Get all the received data
        $rowID = (! empty($data['rowID']) && intval($data['rowID'])) ? intval($data['rowID']) : 0;
        $email = (! empty($data['email'])) ? $data['email'] : '';
        $new = (! empty($data['new'])) ? $data['new'] : '';
        $current = (! empty($data['current'])) ? $data['current'] : '';

        // Process if all the required data are available
        if(! empty($rowID)
            && ! empty($email)
            && ! empty($new)
        ) {

            // Try to get the row that correspond to the specified rowID
            $existingRow = $this->_list($rowID);

            // Check whether the row exist or not
            if(! empty($existingRow)) {

                // Get user random_salt
                $randomSalt = $this->_getDataBy([
                    'email'     => $email,
                    'active'    => ACTIVE,
                    'deleted'   => INJECTION,
                ]);

                // Get encrypted the sent password using the got random_salt
                $password = $this->_hashPassword($current, $randomSalt['random_salt']);

                // Try to get the user that correspond to the specified password
                $userInfos = $this->_getDataBy([
                    'email'     => $email,
                    'pwd'       => $password,
                    'active'    => ACTIVE,
                    'deleted'   => INJECTION,
                ]); // The encrypted one

                // Check whether the encrypted password match any data or not
                if(! empty($userInfos)) {

                    // Generate new random_salt
                    $newRandomSalt = $this->_randomSalt();

                    // Get encrypted the new password using the generated random_salt
                    $newPassword = $this->_hashPassword($new, $newRandomSalt);

                    // Set the data array to update
                    $toUpdateData = [
                        'random_salt'   => $newRandomSalt,
                        'pwd'           => $newPassword,
                        'pwd_update_at' => date('Y-m-d H:i:s'),
                    ];

                    // Update password
                    $updated = $this->update($rowID, $toUpdateData);

                    // Check if the row has been successfully created
                    if($updated) {

                        // Set the informations to return
                        $code = 'success';
                        $message = ucfirst(lang('ForgotPassword.response.updated'));

                    } // End if
                    else {

                        // Set the informations to return
                        $code = 'warning';
                        $message = ucfirst(lang('General.response.server'));

                    } // End else

                } // End if
                else {

                    // Set the informations to return
                    $code = 'danger';
                    $message = ucfirst(lang('Profile.response.incorrectData'));

                } // End else

            } // End if
            else {

                // Set the informations to return
                $code = 'warning';
                $message = ucfirst(lang('Profile.response.notFound'));

            } // End else

        } // End if

        // Set the current function response
        $response = [
            'code'      => $code,
            'message'   => $message,
            'data'      => $responseData,
        ];

        // Return the response
        return $response;

    } // end::_setPassword



    /**
     * FORMAT ROW DATA
     * 
     * @param   array  $data
     * 
     * @return  array  $formatedData
     */
    public function _formatRowData($data = [], $isListPart = false, $validOnly = false)
    {

        // Set the default formatted data to return
        $formattedData = [];

        // Check wether the data to format is empty or not
        if(! empty($data))
        {

            // Get all the formatted data
            $id = (! empty($data['id']) && intval($data['id']) > 0) ? intval($data['id']) : 0;
            $roleID = (! empty($data['role_id']) && intval($data['role_id']) > 0) ? intval($data['role_id']) : 0;
            $roleName = (! empty($data['role_name'])) ? trim($data['role_name']) : '';
            $firstName = (! empty($data['first_name'])) ? trim($data['first_name']) : '';
            $lastName = (! empty($data['last_name'])) ? trim($data['last_name']) : '';
            $phone = (! empty($data['phone'])) ? trim($data['phone']) : '';
            $email = (! empty($data['email'])) ? trim($data['email']) : '';
            $active = (! empty($data['active']) && intval($data['active']) > 0) ? intval($data['active']) : 0;
            $deleted = (! empty($data['deleted']) && intval($data['deleted']) > 0) ? intval($data['deleted']) : 0;
            $created_by = (! empty($data['created_by']) && intval($data['created_by']) > 0) ? intval($data['created_by']) : null;
            $deleted_by = (! empty($data['deleted_by']) && intval($data['deleted_by']) > 0) ? intval($data['deleted_by']) : null;
            $created_at = (! empty($data['created_at'])) ? trim($data['created_at']) : '';
            $updated_at = (! empty($data['updated_at'])) ? trim($data['updated_at']) : null;
            $deleted_at = (! empty($data['deleted_at'])) ? trim($data['deleted_at']) : null;

            // Check if only valid data are requested
            if($validOnly
                && (empty($id) || empty($name))
            )
            {

                // Exit the process
                return $formattedData;

            } // End if

            // Set the formatted data
            $formattedData = [
                'id'            => $id,
                'firstName'     => $firstName,
                'lastName'      => $lastName,
                'phone'         => $phone,
                'email'         => $email,
                'role'          => [
                    'id'    => $roleID,
                    'name'  => $roleName,
                ],
                'active'        => $active,
                'deleted'       => $deleted,
                'created_by'    => $created_by,
                'created_at'    => $created_at,
            ];

            // Check if data are required for a rows list
            if($isListPart)
            {

                // Set the formatted data to return
                return $formattedData;

            } // End if

            // Set the formatted data
            $formattedData['deleted_by']    = $deleted_by;
            $formattedData['updated_at']    = $updated_at;
            $formattedData['deleted_at']    = $deleted_at;

        } // End if

        return $formattedData;

    } // end::_formatRowData



    /**
     * FORMAT ROWS
     * 
     * @param   array  $data
     * 
     * @return  array  $formatedData
     */
    public function _formatRows($data = [], $isListPart = false, $validOnly = false)
    {

        // Set the default formatted data to return
        $formattedData = [];

        // Check wether the data to format is empty or not
        if(! empty($data))
        {

            // Loop through all the data then format each
            foreach($data as $row)
            {

                // Add the row the formatted data array
                $formattedData[] = $this->_formatRowData($row, $isListPart, $validOnly);


            } // End loop

        } // End if

        return $formattedData;

    } // end::_formatRows

} // end::UsersModel class