<?php
namespace App\Models;

// Call Model Namespace
use CodeIgniter\Model;

// Load required models
use App\Models\RolePermissionsModel; // role permissions

// Begin ModeratorsModel class
class ModeratorsModel extends Model
{

    // Defined all the required data
    protected $table = 'moderator';
    protected $primaryKey = 'id';

    protected $returnType     = 'array';
    protected $useSoftDeletes = true;
    
    // Allowed fields for insertion
    protected $allowedFields = ['id', 'role_id', '', 'first_name', 'last_name', 'phone', 'email', 'pwd', 'random_salt', 'photo', 'active', 'deleted', 'created_by', 'deleted_by', 'created_at', 'deleted_at', 'updated_at'];

    protected $searchFields = ['first_name', 'last_name', 'phone', 'email', 'created_at'];
    protected $sortingFields = [
        'first_name'    => 'first_name',
        'last_name'     => 'last_name',
        'phone'         => 'phone',
        'email'         => 'email',
        'active'        => 'active',
        'created_at'    => 'created_at',
    ];

    // Set the allowed field to use as login credentials
    protected $allowedLogin = ['email', 'phone'];

    protected $useTimestamps = false;
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    protected $validationRules    = [];
    protected $validationMessages = [];
    protected $skipValidation     = false;


    

    //-----------------------------------------------------------------
    // LIST ALL THE DATA AVAILABLES OR RETURN A SPECIFIED DATA DETAILS
    //-----------------------------------------------------------------
        public function list($id = 0)
        {

            // Check wether an id has been specified or not
            if ($id === 0)
            {

                // Returns all the rows availables
                // return $this->findAll();
                return $this->select('moderator.id, moderator.first_name, moderator.last_name, moderator.phone, moderator.photo, moderator.email, moderator.active, role.name AS roleName, role.id AS roleID')
                            ->join('role', 'role.id = moderator.role_id', 'left') // Role_id can be null
                            ->asArray()
                            ->where([
                                // 'moderator.active'        => 1,
                                'moderator.deleted'       => 0,
                                'moderator.deleted_by'    => null,
                            ])
                            ->get()
                            ->getResultArray();

            } // End if

            // Return the row details
            return $this->asArray()
                        ->where([
                            'moderator.id' => $id
                        ])
                        ->first(); // Return the first item

        } // End function
    //-----------------------------------------------------------------

    //------------------------------
    // CREATE A NEW ROW IN DATABASE
    //------------------------------
        /**
         * A method that will attempt to
         * add a new line to the `moderator` table
         * and return a boolean on the process result
         *
         * @param array|object $data
         *
         * @return boolean
         */
        public function createRow($data = [])
        {

            // Process only if the sent data is not empty
            if(! empty($data))
            {

                // Get the role_id
                $roleID = (! empty($data['role']) && intval($data['role']) > 0) ? intval($data['role']) : null;

                // Get the required informations for password encryption
                $passwd = $data['pwd'];
                $randomSalt = $this->_randomSalt();
                $encryptedPasswd = $this->hashPasswd($passwd, $randomSalt);

                return $this->save([
                    'role_id'       => $roleID,
                    'first_name'    => $data['firstName'],
                    'last_name'     => $data['lastName'],
                    'phone'         => $data['phone'],
                    'email'         => $data['email'],
                    'random_salt'   => $randomSalt,
                    'pwd'           => $encryptedPasswd,
                    'active'        => $data['active'],
                    'created_by'    => $data['created_by'],
                ]);

            } // End if

            // Return the function response
            return false;

        } // End function
    //------------------------------

    //-----------------------------------------------------
    // ACCESS AND RETURN A USER DATA USING HIS CREDENTIALS
    //-----------------------------------------------------
        public function signIn($credentials = [])
        {

            // Set the default data to return
            $data = [];
            $status = 'denied';
            $message = 'Incorrect login or password provided';

            // Process only if the sent credentials is not empty
            if(! empty($credentials))
            {

                // Get the moderator credentials
                $login = (is_array($credentials) && ! empty($credentials['login'])) ? $credentials['login'] : $credentials->{'login'};
                $passwd = (is_array($credentials) && ! empty($credentials['passwd'])) ? $credentials['passwd'] : $credentials->{'passwd'};

                // Check wether the specified login match any moderator login or not
                if($this->_isUserLogin($login))
                {

                    // Get the moderator random_salt value
                    $randomSalt = $this->_getRandomSaltByLogin($login);

                    // Get encrypted the sent password using the get moderator random_salt
                    $password = $this->hashPasswd($passwd, $randomSalt['random_salt']);

                    // Try to get the moderator that corresponds to the specified password
                    $moderatorInfos = $this->_getUserInfoByPassword($password, $login); // The encrypted one

                    // Check wether the encrypted password match the moderator one or not
                    if(! empty($moderatorInfos))
                    {

                        // Set the data value to return
                        $data = $moderatorInfos;

                        // Load the RolePermissions model
                        $rolePermissionsModel = new RolePermissionsModel();

                        // Try to get the user access
                        $data['permissions'] = $rolePermissionsModel->getRolePermissions($moderatorInfos['roleID']);

                        // Set the new authentification status
                        $status = 'granted';

                        // Set the message to return
                        $message = 'A moderator account successfully matches the sent credentials';

                    } // End if

                } // End if

            } // End if

            // Set the response value
            $response = [
                'status'    => $status,
                'data'      => $data,
                'message'   => $message
            ];

            // Return the function response
            return $response;

        } // End function
    //-----------------------------------------------------

    //------------------------------------------------
    // ACTIVATE THE USER ACCOUNT AND CONNECT THE USER
    //------------------------------------------------
        public function activateAccount($data = [])
        {

            // Set the default data to return
            $responseData = [];
            $status = 'denied';
            $message = 'Invalid or wrong activation code provided';

            // Process only if the sent data is not empty
            if(! empty($data))
            {

                // Get the moderator data
                $email = (is_array($data) && ! empty($data['email'])) ? $data['email'] : $data->{'email'};
                $code = (is_array($data) && ! empty($data['code'])) ? $data['code'] : $data->{'code'};

                // Check wether the specified email match any moderator login or not
                if($this->_isUserNotActivated($email))
                {

                    // Get activated the moderator account
                    $activatedAccount = $this->_activateAccount($data);

                    // Check wether the encrypted password match the moderator one or not
                    if(! empty($activatedAccount))
                    {

                        // Return the moderator activation informations
                        $status = 'granted';
                        $message = 'User account successfully activated';
                        $responseData = $activatedAccount;

                    } // End if

                } // End if

            } // End if

            // Set the response value
            $response = [
                'status'    => $status,
                'data'      => $responseData,
                'message'   => $message
            ];

            // Return the function response
            return $response;

        } // End function
    //------------------------------------------------




    //--------------------------------------------
    // GET A USER INFORMATIONS USING HIS PASSWORD
    //--------------------------------------------
        public function _getUserInfoByPassword($password = '', $login = '')
        {

            // Process only if the encrypted password is not empty
            if(! empty($password)
                && ! empty($login)
            )
            {

                // Get the login array to look for
                $loginArray = $this->_searchArray($login, $this->allowedLogin);

                // Return the function response
                return $this->select('moderator.id, moderator.first_name, moderator.last_name, moderator.phone, moderator.photo, moderator.email, moderator.active, role.name AS roleName, role.id AS roleID')
                            ->join('role', 'role.id = moderator.role_id', 'left') // Role_id can be null
                            ->asArray()
                            ->where('pwd', $password)
                            ->where([
                                'moderator.deleted'     => 0,
                                'moderator.deleted_by'  => null,
                            ])
                            ->groupStart()
                                ->orWhere($loginArray)
                            ->groupEnd()
                            ->first();

                // Return the function response

            } // End if

            // Return the function response
            return null;

        } // End function
    //--------------------------------------------

    //---------------------------------
    // GET A USER PROFILE INFORMATIONS
    //---------------------------------
        /**
         * Return all the moderator profil informations
         * Using the moderator row ID to get data
         *
         * @param UNSIGNED int(10) $moderatorID
         *
         * @return array $data
         */
        public function _getUserProfile($moderatorID = 0)
        {

            // Process only if the encrypted password is not empty
            if(! empty($moderatorID)
                && intval($moderatorID) > 0
            )
            {

                // Return the function response
                return $this->select('moderator.id, moderator.first_name, moderator.last_name, moderator.phone, moderator.photo, moderator.email, moderator.active, role.name AS roleName, role.id AS roleID')
                            ->join('role', 'role.id = moderator.role_id', 'left') // Role_id can be null
                            ->asArray()
                            ->where('moderator.id', $moderatorID)
                            ->where([
                                'moderator.deleted'     => 0,
                                'moderator.deleted_by'  => null,
                            ])
                            ->first();

            } // End if

            // Return the function response
            return null;

        } // End function
    //---------------------------------

    //----------------------------------------------
    // GET A USER RANDOM_SALT VALUE USING HIS LOGIN
    //----------------------------------------------
        public function _getRandomSaltByLogin($login = '')
        {

            // Process only if the login is not empty
            if(! empty($login))
            {

                // Get the login array to look for
                $loginArray = $this->_searchArray($login, $this->allowedLogin);

                // Process only if there is any data to look for
                if(! empty($loginArray))
                {

                    // Return the function response
                    return $this->select('random_salt')
                                ->asArray()
                                ->where([
                                    'deleted'       => 0,
                                    'deleted_by'    => null,
                                ])
                                ->groupStart()
                                    ->orWhere($loginArray)
                                ->groupEnd()
                                ->first();

                } // End if

            } // End if

            // Return the function response
            return false;

        } // End function
    //----------------------------------------------

    //---------------------------------------------------------------
    // CHECK IF A SPECIFIED LOGIN MATCH ANY USER LOGIN FROM DATABASE
    //---------------------------------------------------------------
        public function _isUserLogin($login = '')
        {

            // Process only if the login is not empty
            if(! empty($login))
            {

                // Get the login array to look for
                $loginArray = $this->_searchArray($login, $this->allowedLogin);

                // Process only if there is any data to look for
                if(! empty($loginArray))
                {

                    // Get the found moderator
                    $moderatorId = $this->select('id')
                                ->asArray()
                                ->where([
                                    'deleted'       => 0,
                                    'deleted_by'    => null,
                                ])
                                ->groupStart()
                                    ->orWhere($loginArray)
                                ->groupEnd()
                                ->first();

                    // Return the function response
                    return (! empty($moderatorId)) ? true : false;

                } // End if

            } // End if

            // Return the function response
            return false;

        } // End function
    //---------------------------------------------------------------

    //------------------------------------------------------------
    // CHECK IF A SPECIFIED EMAIL MATCH ANY INACTIVE USER ACCOUNT
    //------------------------------------------------------------
        public function _isUserNotActivated($email = '')
        {

            // Process only if the email is not empty
            if(! empty($email))
            {

                // Get the found moderator
                $moderatorId = $this->select('id')
                            ->asArray()
                            ->where([
                                'email'     => $email,
                                'active'    => 0,
                            ])
                            ->first();

                // Return the function response
                return (! empty($moderatorId)) ? true : false;

            } // End if

            // Return the function response
            return false;

        } // End function
    //------------------------------------------------------------

    //------------------------------------------
    // CHECK WETHER A ROW IS A DUPLICATE OR NOT
    //------------------------------------------
        public function isDuplicate($where = [], $orWhere = [])
        {

            // Set the default answer to return
            $isDuplicate = true;

            // Try to get the row
            $row = $this->where($where)
                        ->where([
                            'deleted'       => 0,
                            'deleted_by'    => null,
                        ])
                        ->groupStart()
                            ->orWhere($orWhere)
                        ->groupEnd()
                        ->get()
                        ->getResultArray();

            // Check the row
            if( empty($row))
            {

                // Set the row as not duplicate
                $isDuplicate = false;

            } // End if

            // Return the answer
            return $isDuplicate;

        } // End function
    //------------------------------------------

    //-------------------------------
    // COUNT ALL THE ROWS AVAILABLES
    //-------------------------------
        public function countTotal($keyword = '', $allFields = true)
        {

            // Process only if the keyword is not empty
            // Search through all the availables fields if required
            if($allFields && ! empty($keyword))
            {

                // Get the search array
                $searchArray = $this->_searchArray($keyword);

                // Return the rows that match any column
                return $this->orLike($searchArray)
                            ->countAllResults();

            } // End if

            // Return the rows that match even if the keyword is empty
            return $this->like($this->searchFields[0], $keyword)
                        ->countAllResults();

        } // End function
    //-------------------------------

    //-------------------------------------------
    // LIST ALL THE ROWS THAT MATCHS THE KEYWORD
    //-------------------------------------------
        public function search($keyword = '', $allFields = true, $options = [])
        {

            // Get any pagination option specified
            $paging = (! empty($options) && ! empty($options['paging'])) ? $options['paging'] : [];
            $page = (! empty($paging) && ! empty($paging['page']) && intval($paging['page']) > 0) ? intval($paging['page']) : 0;
            $length = (! empty($paging) && ! empty($paging['length']) && intval($paging['length']) > 0) ? intval($paging['length']) : 20;
            $sorting = (! empty($options) && ! empty($options['sorting'])) ? $options['sorting'] : [];
            $field = (! empty($sorting) && ! empty($sorting['field'])) ? mb_strtolower($sorting['field']) : '';
            $sort = (! empty($sorting) && ! empty($sorting['sort']) && mb_strtoupper($sorting['sort']) === 'ASC') ? 'ASC' : 'DESC';

            // Adjust the query if the paging option is specified
            if(! empty($page))
            {

                // Set the offset value
                $offset = ($page - 1) * $length;

                // Add paging option
                $this->limit($length, $offset);

            } // End if

            // Adjust the query if the sorting option is specified
            if(! empty($field)
                && array_key_exists($field, $this->sortingFields)
            )
            {

                // Add sorting option
                $this->orderBy($this->sortingFields[$field], $sort);

            } // End if

            // Search through all the availables fields if required
            if($allFields && ! empty($keyword))
            {

                // Get the search array
                $searchArray = $this->_searchArray($keyword);

                // Return the rows that match any column
                return $this->orLike($searchArray)
                            ->get()
                            ->getResultArray();

            } // End if

            // Process only if the keyword is not empty
            return $this->like($this->searchFields[0], $keyword)
                        ->get()
                        ->getResultArray();

        } // End function
    //-------------------------------------------

    //---------------------------------------------------------
    // RETURN AN ARRAY OF COLUMN WHERE TO LOOK FOR THE KEYWORD
    //---------------------------------------------------------
        public function _searchArray($keyword = '', $searchFields = [])
        {

            // Set the default data to return
            $data = [];

            // Get the array to retrieve
            $searchFields = (! empty($searchFields)) ? $searchFields : $this->searchFields;

            // Loop through all the fields then add the keyword as value
            foreach($searchFields as $searchField):
                
                // Set a new array item
                $data["$searchField"] = $keyword;

            endforeach; // End loop

            // Return data as function response
            return $data;


        } // End function
    //---------------------------------------------------------

    //----------------------------
    // GENERATE A RANDOM SALT KEY
    //----------------------------
        public function _randomSalt()
        {

            // Return the key
            return md5(mt_rand());

        } // End function
    //----------------------------

    //-----------------------------------------
    // HASH A PASSWORD USING A RANDOM SALT KEY
    //-----------------------------------------
        public function hashPasswd($passwd, $randomSalt)
        {

            /**
             * bcrypt is the preferred hashing for passwords, but
             * is only available for PHP 5.3+. Even in a PHP 5.3+ 
             * environment, we have the option to use PBKDF2; just 
             * set the PHP52_COMPATIBLE_PASSWORDS constant located 
             * in config/constants.php to 1.
             */

            // Load the Encryption Library
            $encrypter = \Config\Services::encrypter();

            // Return the encrypted password
            return crypt($passwd . $encrypter->key, '$2a$09$' . $randomSalt . '$');

        } // End function
    //-----------------------------------------

} // End ModeratorsModel class