<?php

/* Namespace the app controller */
namespace App\Controllers;

/* Here to load required models */
use App\Models\RolesModel; // Roles
use App\Models\PermissionsModel; // Permissions
use App\Models\UsersModel; // Users


// Import PHPMailer classes into the global namespace
// These must be at the top of your script, not inside a function
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

// Require the Autoload file
require_once COMPOSER_PATH;

// begin::Settings
class Settings extends BaseController
{


    /**
     * INDEX METHOD
     * FIRST TO BE LOADED
     */
    public function index()
    {

        // Check wether the user is already connected or not
        if(empty($this->session->connected))
        {

            // Redirect the user to the default controller
            return redirect()->to(base_url($this->data['locale'] . '/sign/out'));

        } // End if

        // Load the roles method as default one
        $this->roles();

    } // end::index

    

    /**
     * LIST USERS
     */
    public function users()
    {

        // Check wether the user is already connected or not
        if(empty($this->session->connected))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/sign/out'));

        } // End if


        /* Localization */
        $usersLanguagePath = "Language/{$this->data['locale']}/Users.php";
        $usersLanguageFile = (file_exists(APPPATH . $usersLanguagePath)) ? require_once APPPATH . $usersLanguagePath : '';


        // Check wether the user is already connected or not
        if(! $this->_hasPermission('READ_USER'))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/dashboard'));

        } // End if

        // Set all the page scripts to load
        $this->data['pageScripts']  = [

            // Routes datatable file script
            'crud/project-datatables/users-datatable.js',

        ];

        // Specify the menu to highlight
        $this->data['activeMenu']['settings']['users']  = 'menu-item-active';

        // Set the page details
        $this->data['pageDetails']  = [
            'title'         => ucfirst(lang('Users.title')),
            'description'   => ucfirst(lang('Users.description')),
            'localization'  => [
                'users' => $usersLanguageFile,
            ],
        ];

        // Render the page content that corresponds
        $this->template->render('settings/users/list', $this->data);

    } // end::users
    
    /**
     * GET USERS DATA
     * IN JSON FORMAT
     * 
     * @param   array  $_POST
     * 
     * @return  JSON   $response
     */
    public function getUsersJSON($search = '')
    {

        // Get all the sent data
        $pagination = (! empty($this->request->getPost('pagination'))) ? $this->request->getPost('pagination') : [];
        $sorting = (! empty($this->request->getPost('sort'))) ? $this->request->getPost('sort') : [];
        $query = (! empty($this->request->getPost('query'))) ? $this->request->getPost('query') : [];
        $page = (! empty($pagination) && ! empty($pagination['page'])) ? intval($pagination['page']) : 1;
        $perPage = (! empty($pagination) && ! empty($pagination['perpage'])) ? intval($pagination['perpage']) : 10;
        $field = (! empty($sorting) && ! empty($sorting['field'])) ? $sorting['field'] : '';
        $sort = (! empty($sorting) && ! empty($sorting['sort'])) ? $sorting['sort'] : '';
        $search = (! empty($query) && ! empty($query['generalSearch'])) ? trim($query['generalSearch']) : $search;
        $status = (isset($query['active']) && (! empty($query['active']) || $query['active'] === '0')) ? intval($query['active']) : 'all';


        // Set the paging options
        $paging = [
            'page'      => $page,
            'length'    => $perPage,
        ];

        // Set the sorting options
        $sortingOptions = [
            'field' => $field,
            'sort'  => $sort,
        ];

        // Load the required models
        $usersModel = new UsersModel(); // Users

        // Get the requested data
        $data = $usersModel->list(0, [
            'search'    => $search,
            'status'    => $status,
            'paging'    => $paging,
            'sorting'   => $sortingOptions,
        ]);

        // Set the response
        $response = [
            'meta'  => $data['meta'],
            'data'  => $data['data'],
        ];

        // Return the function response
        return $this->response->setJSON($response);

    } // end::getUsersJSON

    /**
     * USER FORM
     */
    public function userForm()
    {

        // Get all the required data
        $rowID = (! empty($this->request->getPost()) && ! empty($this->request->getPost('rowID'))) ? $this->request->getPost('rowID') : 0;

        // Load the required models
        $usersModel = new UsersModel(); // Users
        $rolesModel = new RolesModel(); // Roles

        // Load all the data list required
        $roles = $rolesModel->asOptions();

        // Get the row details
        $rowDetails = (! empty($rowID)) ? $usersModel->list($rowID) : [];
        $rowData = (! empty($rowDetails)) ? $rowDetails['data'] : [];

        // Set all the data to sent to the form
        $formData = [
            'action'    => base_url($this->data['locale']) . '/user/set',
            'row'       => $rowData,
            'data'      => [
                'roles'   => $roles,
            ],
        ];

        // Render the view that corresponds
        print view('settings/users/form', $formData);

    } // end::userForm

    /**
     * SET USER
     * 
     * @param   array  $_POST
     * 
     * @return  array  $response
     */
    public function setUser()
    {

        // Get all the posted data
        $rowID = (! empty($this->request->getPost('row-ID')) && intval($this->request->getPost('row-ID')) > 0) ? intval($this->request->getPost('row-ID')) : 0;
        $firstName = (! empty($this->request->getPost('txt-first-name'))) ? trim($this->request->getPost('txt-first-name')) : '';
        $lastName = (! empty($this->request->getPost('txt-last-name'))) ? trim($this->request->getPost('txt-last-name')) : '';
        $email = (! empty($this->request->getPost('txt-email'))) ? trim($this->request->getPost('txt-email')) : '';
        $phone = (! empty($this->request->getPost('txt-phone'))) ? trim($this->request->getPost('txt-phone')) : '';
        $role = (! empty($this->request->getPost('slt-role')) && intval($this->request->getPost('slt-role')) > 0) ? intval($this->request->getPost('slt-role')) : 0;
        $password = $this->_generatePassword();
        // $randomSalt = $this->_randomSalt();

        // Set the data to process
        $data = [
            'firstName'     => $firstName,
            'lastName'      => $lastName,
            'email'         => $email,
            'phone'         => $phone,
            'role'          => $role,
            'password'      => $password,
            // 'randomSalt'    => $randomSalt,
            // 'randomSalt'    => $randomSalt,
        ];

        // Load the required models
        $usersModel = new UsersModel(); // Users

        // Try to authenticate the user
        $response = $usersModel->setRow($rowID, $data);

        // Process only if there is any user informations returned
        if($response['code'] === 'success') {

            /**
             * PHPMAILER MAIL
             */
            // Instantiation and passing `true` enables exceptions
            $mail = new PHPMailer(true);

            //Server settings
            $mail->SMTPDebug = 0;                               //Enable verbose debug output
            $mail->isSMTP();                                    //Send using SMTP
            $mail->Host       = 'mail.xsel-ci.com';             //Set the SMTP server to send through
            $mail->SMTPAuth   = true;                           //Enable SMTP authentication
            $mail->Username   = 'admin@xsel-ci.com';            //SMTP username
            $mail->Password   = 'pwdAdmin00';                   //SMTP password
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;    //Enable TLS encryption;
            $mail->Port       = 465;                            //use 465 for `PHPMailer::ENCRYPTION_SMTPS` above

            //Recipients
            $mail->setFrom('admin@xsel-ci.com', 'BTP PRO - SIA INGENIERIE');
            $mail->addAddress($email);                          // Add a recipient
            $mail->addReplyTo('admin@xsel-ci.com', 'BTP PRO - SIA INGENIERIE');

            //Content
            $mail->isHTML(true);                                //Set email format to HTML
            $mail->CharSet = 'UTF-8';
            $mail->Subject = 'Bienvenue';

            // Set the HTML message
            $HTMLContent = $this->view->setData(['data' => [
                'firstName' => $firstName,
                'lastName'  => $lastName,
                'email'     => $email,
                'phone'     => $phone,
                'pwd'       => $password,
                'dashboard' => '',                              // Empty for default base_url location
            ]])->render('mailing/welcome');
            $mail->Body    = $HTMLContent;

            // Send the mail
            $mail->send();

        } // End if

        // Return the function response
        return $this->response->setJSON($response);

    } // end::setUser

    /**
     * UDPATE ROW STATUS
     * ENABLE / DISABLE
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function statusUser()
    {

        // Get all the sent form's informations
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;
        $status = (! empty($this->request->getPost('status')) && intval($this->request->getPost('status')) > 0) ? intval($this->request->getPost('status')) : 0;

        // Load the required models
        $usersModel = new UsersModel(); // Users Model

        // Set the response to return
        $response = $usersModel->setStatus($rowID, $status);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::statusUser

    /**
     * DELETE ROW
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function deleteUser()
    {

        // Get all the received data
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;

        // Load the required models
        $permissionsModel = new UsersModel(); // Users Model

        // Set the response to return
        $response = $permissionsModel->deleteRow($rowID);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::deleteUser

    

    /**
     * LIST PERMISSIONS
     */
    public function permissions()
    {

        // Check wether the user is already connected or not
        if(empty($this->session->connected))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/sign/out'));

        } // End if


        /* Localization */
        $permissionsLanguagePath = "Language/{$this->data['locale']}/Permissions.php";
        $permissionsLanguageFile = (file_exists(APPPATH . $permissionsLanguagePath)) ? require_once APPPATH . $permissionsLanguagePath : '';


        // Check wether the user is already connected or not
        if(! $this->_hasPermission('READ_PERMISSION'))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/dashboard'));

        } // End if

        // Set all the page scripts to load
        $this->data['pageScripts']  = [

            // Routes datatable file script
            'crud/project-datatables/permissions-datatable.js',

        ];

        // Specify the menu to highlight
        $this->data['activeMenu']['settings']['permissions']  = 'menu-item-active';

        // Set the page details
        $this->data['pageDetails']  = [
            'title'         => ucfirst(lang('Permissions.title')),
            'description'   => ucfirst(lang('Permissions.description')),
            'localization'  => [
                'permissions' => $permissionsLanguageFile,
            ],
        ];

        // Render the page content that corresponds
        $this->template->render('settings/permissions/list', $this->data);

    } // end::permissions
    
    /**
     * GET PERMISSIONS DATA
     * IN JSON FORMAT
     * 
     * @param   array  $_POST
     * 
     * @return  JSON   $response
     */
    public function getPermissionsJSON($search = '')
    {

        // Get all the sent data
        $pagination = (! empty($this->request->getPost('pagination'))) ? $this->request->getPost('pagination') : [];
        $sorting = (! empty($this->request->getPost('sort'))) ? $this->request->getPost('sort') : [];
        $query = (! empty($this->request->getPost('query'))) ? $this->request->getPost('query') : [];
        $page = (! empty($pagination) && ! empty($pagination['page'])) ? intval($pagination['page']) : 1;
        $perPage = (! empty($pagination) && ! empty($pagination['perpage'])) ? intval($pagination['perpage']) : 10;
        $field = (! empty($sorting) && ! empty($sorting['field'])) ? $sorting['field'] : '';
        $sort = (! empty($sorting) && ! empty($sorting['sort'])) ? $sorting['sort'] : '';
        $search = (! empty($query) && ! empty($query['generalSearch'])) ? trim($query['generalSearch']) : $search;
        $status = (isset($query['active']) && (! empty($query['active']) || $query['active'] === '0')) ? intval($query['active']) : 'all';


        // Set the paging options
        $paging = [
            'page'      => $page,
            'length'    => $perPage,
        ];

        // Set the sorting options
        $sortingOptions = [
            'field' => $field,
            'sort'  => $sort,
        ];

        // Load the required models
        $permissionsModel = new PermissionsModel(); // Permissions

        // Get the requested data
        $data = $permissionsModel->list(0, [
            'search'    => $search,
            'status'    => $status,
            'paging'    => $paging,
            'sorting'   => $sortingOptions,
        ]);

        // Set the response
        $response = [
            'meta'  => $data['meta'],
            'data'  => $data['data'],
        ];

        // Return the function response
        return $this->response->setJSON($response);

    } // end::getPermissionsJSON

    /**
     * PERMISSION FORM
     */
    public function permissionForm()
    {

        // Get all the required data
        $rowID = (! empty($this->request->getPost()) && ! empty($this->request->getPost('rowID'))) ? $this->request->getPost('rowID') : 0;

        // Load the required models
        $permissionsModel = new PermissionsModel(); // Permissions

        // Get the row details
        $rowDetails = (! empty($rowID)) ? $permissionsModel->list($rowID) : [];
        $rowData = (! empty($rowDetails)) ? $rowDetails['data'] : [];

        // Set all the data to sent to the form
        $formData = [
            'action'    => base_url($this->data['locale']) . '/permission/set',
            'row'       => $rowData,
            'data'      => [],
        ];

        // Render the view that corresponds
        print view('settings/permissions/form', $formData);

    } // end::permissionForm

    /**
     * SET PERMISSION
     * 
     * @param   array  $_POST
     * 
     * @return  array  $response
     */
    public function setPermission()
    {

        // Get all the posted data
        $rowID = (! empty($this->request->getPost('row-ID')) && intval($this->request->getPost('row-ID')) > 0) ? intval($this->request->getPost('row-ID')) : 0;
        $name = (! empty($this->request->getPost('txt-name'))) ? trim($this->request->getPost('txt-name')) : '';
        $code = (! empty($this->request->getPost('txt-code'))) ? trim($this->request->getPost('txt-code')) : '';
        $description = (! empty($this->request->getPost('txt-description'))) ? trim($this->request->getPost('txt-description')) : '';

        // Set the data to process
        $data = [
            'name'          => $name,
            'code'          => $code,
            'description'   => $description,
        ];

        // Load the required models
        $permissionsModel = new PermissionsModel(); // Permissions

        // Try to authenticate the user
        $response = $permissionsModel->setRow($rowID, $data);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::setPermission

    /**
     * UDPATE ROW STATUS
     * ENABLE / DISABLE
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function statusPermission()
    {

        // Get all the sent form's informations
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;
        $status = (! empty($this->request->getPost('status')) && intval($this->request->getPost('status')) > 0) ? intval($this->request->getPost('status')) : 0;

        // Load the required models
        $permissionsModel = new PermissionsModel(); // Permissions Model

        // Set the response to return
        $response = $permissionsModel->setStatus($rowID, $status);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::statusPermission

    /**
     * DELETE ROW
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function deletePermission()
    {

        // Get all the received data
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;

        // Load the required models
        $permissionsModel = new PermissionsModel(); // Permissions Model

        // Set the response to return
        $response = $permissionsModel->deleteRow($rowID);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::deletePermission

    

    /**
     * LIST ROLES
     */
    public function roles()
    {

        // Check wether the user is already connected or not
        if(empty($this->session->connected))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/sign/out'));

        } // End if


        /* Localization */
        $rolesLanguagePath = "Language/{$this->data['locale']}/Roles.php";
        $rolesLanguageFile = (file_exists(APPPATH . $rolesLanguagePath)) ? require_once APPPATH . $rolesLanguagePath : '';


        // Check wether the user is already connected or not
        if(! $this->_hasPermission('READ_ROLE'))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/dashboard'));

        } // End if

        // Set all the page scripts to load
        $this->data['pageScripts']  = [

            // Routes datatable file script
            'crud/project-datatables/roles-datatable.js',

        ];

        // Specify the menu to highlight
        $this->data['activeMenu']['settings']['roles']  = 'menu-item-active';

        // Set the page details
        $this->data['pageDetails']  = [
            'title'         => ucfirst(lang('Roles.title')),
            'description'   => ucfirst(lang('Roles.description')),
            'localization'  => [
                'roles' => $rolesLanguageFile,
            ],
        ];

        // Render the page content that corresponds
        $this->template->render('settings/roles/list', $this->data);

    } // end::roles

    /**
     * GET ROLES DATA
     * IN JSON FORMAT
     * 
     * @param   array  $_POST
     * 
     * @return  JSON   $response
     */
    public function getRolesJSON($search = '')
    {

        // Get all the sent data
        $pagination = (! empty($this->request->getPost('pagination'))) ? $this->request->getPost('pagination') : [];
        $sorting = (! empty($this->request->getPost('sort'))) ? $this->request->getPost('sort') : [];
        $query = (! empty($this->request->getPost('query'))) ? $this->request->getPost('query') : [];
        $page = (! empty($pagination) && ! empty($pagination['page'])) ? intval($pagination['page']) : 1;
        $perPage = (! empty($pagination) && ! empty($pagination['perpage'])) ? intval($pagination['perpage']) : 10;
        $field = (! empty($sorting) && ! empty($sorting['field'])) ? $sorting['field'] : '';
        $sort = (! empty($sorting) && ! empty($sorting['sort'])) ? $sorting['sort'] : '';
        $search = (! empty($query) && ! empty($query['generalSearch'])) ? trim($query['generalSearch']) : $search;
        $status = (isset($query['active']) && (! empty($query['active']) || $query['active'] === '0')) ? intval($query['active']) : 'all';


        // Test
        // ...
        // var_dump($status);
        // exit;


        // Set the paging options
        $paging = [
            'page'      => $page,
            'length'    => $perPage,
        ];

        // Set the sorting options
        $sortingOptions = [
            'field' => $field,
            'sort'  => $sort,
        ];

        // Load the required models
        $rolesModel = new RolesModel(); // Roles

        // Get the requested data
        $data = $rolesModel->list(0, [
            'search'    => $search,
            'status'    => $status,
            'paging'    => $paging,
            'sorting'   => $sortingOptions,
        ]);

        // Set the response
        $response = [
            'meta'  => $data['meta'],
            'data'  => $data['data'],
        ];

        // Return the function response
        return $this->response->setJSON($response);

    } // end::getRolesJSON

    /**
     * ROLE FORM
     */
    public function roleForm()
    {

        // Get all the required data
        $rowID = (! empty($this->request->getPost()) && ! empty($this->request->getPost('rowID'))) ? $this->request->getPost('rowID') : 0;

        // Load the required models
        $rolesModel = new RolesModel(); // Roles
        $permissionsModel = new PermissionsModel(); // Permissions

        // Load all the data list required
        $permissions = $permissionsModel->asOptions();

        // Get the row details
        $rowDetails = (! empty($rowID)) ? $rolesModel->list($rowID) : [];
        $rowData = (! empty($rowDetails)) ? $rowDetails['data'] : [];

        // Set all the data to sent to the form
        $formData = [
            'action'    => base_url($this->data['locale']) . '/role/set',
            'row'       => $rowData,
            'data'      => [
                'permissions'   => $permissions,
            ],
        ];

        // Render the view that corresponds
        print view('settings/roles/form', $formData);

    } // end::roleForm

    /**
     * SET ROLE
     * 
     * @param   array  $_POST
     * 
     * @return  array  $response
     */
    public function setRole()
    {

        // Get all the posted data
        $rowID = (! empty($this->request->getPost('row-ID')) && intval($this->request->getPost('row-ID')) > 0) ? intval($this->request->getPost('row-ID')) : 0;
        $name = (! empty($this->request->getPost('txt-name'))) ? trim($this->request->getPost('txt-name')) : '';
        $description = (! empty($this->request->getPost('txt-description'))) ? trim($this->request->getPost('txt-description')) : '';
        $permissions = (! empty($this->request->getPost('slt-permissions'))) ? $this->request->getPost('slt-permissions') : [];

        // Set the data to process
        $data = [
            'name'          => $name,
            'description'   => $description,
            'permissions'   => $permissions,
        ];

        // Load the required models
        $rolesModel = new RolesModel(); // Roles

        // Try to authenticate the user
        $response = $rolesModel->setRow($rowID, $data);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::setRole

    /**
     * UDPATE ROW STATUS
     * ENABLE / DISABLE
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function statusRole()
    {

        // Get all the sent form's informations
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;
        $status = (! empty($this->request->getPost('status')) && intval($this->request->getPost('status')) > 0) ? intval($this->request->getPost('status')) : 0;

        // Load the required models
        $rolesModel = new RolesModel(); // Roles Model

        // Set the response to return
        $response = $rolesModel->setStatus($rowID, $status);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::statusRole

    /**
     * DELETE ROW
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function deleteRole()
    {

        // Get all the received data
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;

        // Load the required models
        $rolesModel = new RolesModel(); // Roles Model

        // Set the response to return
        $response = $rolesModel->deleteRow($rowID);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::deleteRole

    

    /**
     * LIST CURRENCIES
     */
    public function currencies()
    {

        // Check wether the user is already connected or not
        if(empty($this->session->connected))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/sign/out'));

        } // End if


        /* Localization */
        $currenciesLanguagePath = "Language/{$this->data['locale']}/Currencies.php";
        $currenciesLanguageFile = (file_exists(APPPATH . $currenciesLanguagePath)) ? require_once APPPATH . $currenciesLanguagePath : '';


        // Check wether the user is already connected or not
        if(! $this->_hasPermission('READ_CURRENCY'))
        {

            // Redirect the user to the dashboard controller
            return redirect()->to(base_url($this->data['locale'] . '/dashboard'));

        } // End if

        // Set all the page scripts to load
        $this->data['pageScripts']  = [

            // Routes datatable file script
            'crud/project-datatables/currencies-datatable.js',

        ];

        // Specify the menu to highlight
        $this->data['activeMenu']['settings']['currencies']  = 'menu-item-active';

        // Set the page details
        $this->data['pageDetails']  = [
            'title'         => ucfirst(lang('Currencies.title')),
            'description'   => ucfirst(lang('Currencies.description')),
            'localization'  => [
                'currencies' => $currenciesLanguageFile,
            ],
        ];

        // Render the page content that corresponds
        $this->template->render('settings/currencies/list', $this->data);

    } // end::currencies
    
    /**
     * GET CURRENCIES DATA
     * IN JSON FORMAT
     * 
     * @param   array  $_POST
     * 
     * @return  JSON   $response
     */
    public function getCurrenciesJSON($search = '')
    {

        // Get all the sent data
        $pagination = (! empty($this->request->getPost('pagination'))) ? $this->request->getPost('pagination') : [];
        $sorting = (! empty($this->request->getPost('sort'))) ? $this->request->getPost('sort') : [];
        $query = (! empty($this->request->getPost('query'))) ? $this->request->getPost('query') : [];
        $page = (! empty($pagination) && ! empty($pagination['page'])) ? intval($pagination['page']) : 1;
        $perPage = (! empty($pagination) && ! empty($pagination['perpage'])) ? intval($pagination['perpage']) : 10;
        $field = (! empty($sorting) && ! empty($sorting['field'])) ? $sorting['field'] : '';
        $sort = (! empty($sorting) && ! empty($sorting['sort'])) ? $sorting['sort'] : '';
        $search = (! empty($query) && ! empty($query['generalSearch'])) ? trim($query['generalSearch']) : $search;
        $status = (isset($query['active']) && (! empty($query['active']) || $query['active'] === '0')) ? intval($query['active']) : 'all';


        // Set the paging options
        $paging = [
            'page'      => $page,
            'length'    => $perPage,
        ];

        // Set the sorting options
        $sortingOptions = [
            'field' => $field,
            'sort'  => $sort,
        ];

        // Load the required models
        $currenciesModel = new CurrenciesModel(); // Currencies

        // Get the requested data
        $data = $currenciesModel->list(0, [
            'search'    => $search,
            'status'    => $status,
            'paging'    => $paging,
            'sorting'   => $sortingOptions,
        ]);

        // Set the response
        $response = [
            'meta'  => $data['meta'],
            'data'  => $data['data'],
        ];

        // Return the function response
        return $this->response->setJSON($response);

    } // end::getCurrenciesJSON

    /**
     * CURRENCY FORM
     */
    public function currencyForm()
    {

        // Get all the required data
        $rowID = (! empty($this->request->getPost()) && ! empty($this->request->getPost('rowID'))) ? $this->request->getPost('rowID') : 0;

        // Load the required models
        $currenciesModel = new CurrenciesModel(); // Currencies

        // Get the row details
        $rowDetails = (! empty($rowID)) ? $currenciesModel->list($rowID) : [];
        $rowData = (! empty($rowDetails)) ? $rowDetails['data'] : [];

        // Set all the data to sent to the form
        $formData = [
            'action'    => base_url($this->data['locale']) . '/currency/set',
            'row'       => $rowData,
            'data'      => [],
        ];

        // Render the view that corresponds
        print view('settings/currencies/form', $formData);

    } // end::currencyForm

    /**
     * SET CURRENCY
     * 
     * @param   array  $_POST
     * 
     * @return  array  $response
     */
    public function setCurrency()
    {

        // Get all the posted data
        $rowID = (! empty($this->request->getPost('row-ID')) && intval($this->request->getPost('row-ID')) > 0) ? intval($this->request->getPost('row-ID')) : 0;
        $name = (! empty($this->request->getPost('txt-name'))) ? trim($this->request->getPost('txt-name')) : '';
        $code = (! empty($this->request->getPost('txt-code'))) ? trim($this->request->getPost('txt-code')) : '';
        $description = (! empty($this->request->getPost('txt-description'))) ? trim($this->request->getPost('txt-description')) : '';

        // Set the data to process
        $data = [
            'name'          => $name,
            'code'          => $code,
            'description'   => $description,
        ];

        // Load the required models
        $currenciesModel = new CurrenciesModel(); // Currencies

        // Try to authenticate the user
        $response = $currenciesModel->setRow($rowID, $data);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::setCurrency

    /**
     * UDPATE ROW STATUS
     * ENABLE / DISABLE
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function statusCurrency()
    {

        // Get all the sent form's informations
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;
        $status = (! empty($this->request->getPost('status')) && intval($this->request->getPost('status')) > 0) ? intval($this->request->getPost('status')) : 0;

        // Load the required models
        $currenciesModel = new CurrenciesModel(); // Currencies Model

        // Set the response to return
        $response = $currenciesModel->setStatus($rowID, $status);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::statusCurrency

    /**
     * DELETE ROW
     * 
     * @param   array  $_POST
     * 
     * @return  string $response
     */
    public function deleteCurrency()
    {

        // Get all the received data
        $rowID = (! empty($this->request->getPost('rowID')) && intval($this->request->getPost('rowID')) > 0) ? intval($this->request->getPost('rowID')) : 0;

        // Load the required models
        $currenciesModel = new CurrenciesModel(); // Currencies Model

        // Set the response to return
        $response = $currenciesModel->deleteRow($rowID);

        // Return the function response
        return $this->response->setJSON($response);

    } // end::deleteCurrency


} // end::Settings