<?php

// Namespace to Models folder
namespace App\Models;

// Call BaseModel Namespace
use App\Models\BaseModel;


// begin::PermissionsModel class
class PermissionsModel extends BaseModel
{

    // Model configuration
    protected $table = 'permission';
    protected $primaryKey = 'id';
    
    // Allowed fields for insertion
    protected $allowedFields = ['id', 'name', 'code', 'description', 'active', 'deleted', 'created_by', 'deleted_by', 'created_at', 'deleted_at', 'updated_at'];
    
    // Fields available on SELECT query
    protected $selectedFields = 'permission.id, permission.name, permission.code, permission.description, permission.active, permission.deleted, permission.created_by, permission.deleted_by, permission.created_at, permission.updated_at, permission.deleted_at';

    // Fields used on filters queries
    // Used by _getDataBy() function
    protected $filtersFields = [
        'id'            => 'permission.id',
        'name'          => 'permission.name',
        'code'          => 'permission.code',
        'description'   => 'permission.description',
        'active'        => 'permission.active',
        'deleted'       => 'permission.deleted',
    ];

    // Fields used for SEARCH query
    protected $searchFields = [
        'permission.name',
        'permission.code',
        'permission.description',
        'permission.active',
        'permission.created_at',
    ];

    // Fields available for sorting
    protected $sortingFields = [
        'name'          => 'permission.name',
        'code'          => 'permission.code',
        'description'   => 'permission.description',
        'active'        => 'permission.active',
        'created_at'    => 'permission.created_at',
    ];

    // Set the joins array
    protected $joints = [
    ];



    /**
     * GET DATA LIST
     * 
     * @param   int    $rowID
     * @param   array  $options
     * 
     * @return  array  $response
     */
    public function list(int $rowID = 0, array $options = [])
    {

        // Get the row ID
        $rowID = (! empty($rowID) && intval($rowID) > 0) ? intval($rowID) : 0;

        // Process only if the row ID has been specified
        if($rowID === 0) {

            // Get all the received data
            $paging = (! empty($options['paging'])) ? $options['paging'] : [];
            $sorting = (! empty($options['sorting'])) ? $options['sorting'] : [];
            $search = (! empty($options['search'])) ? trim($options['search']) : '';
            $status = (! empty($options['status']) || $options['status'] === '0') ? intval($options['status']) : 'all';

            // Check if any keyword has been sent for search
            if(! empty($search)) {

                // Load the requested data
                $responseData = $this->_search($search, true, [
                    'status'    => $status,
                    'paging'    => $paging,
                    'sorting'   => $sorting,
                ]);

            } // End if
            else {

                // Load the requested data
                $responseData = $this->_list($rowID, [
                    'status'    => $status,
                    'paging'    => $paging,
                    'sorting'   => $sorting,
                ]);

            } // End else

            // Get the total rows availables
            $total = $this->_countTotal($search);

            // Get the total pages availables
            $pages = ceil($total / $responseData['length']);

            $page = ($responseData['page'] > $pages) ? $pages : $responseData['page'];

            // Set the response to return
            $response = [
                'meta'  => [
                    'page'      => $page,
                    'pages'     => $pages,
                    'perpage'   => $responseData['length'],
                    'total'     => $total,
                    'sort'      => $responseData['sort'],
                    'field'     => $responseData['field'],
                ],
                'data'  => $this->_formatRows($responseData['data'], true),
            ];

        } // End if
        else {

            // Load the requested data
            $responseData = $this->_getDataBy([
                'id'    => $rowID
            ]);

            // Set the response to return
            $response = [
                'meta'  => [],
                'data'  => $this->_formatRowData($responseData),
            ];

        } // End else

        // Return the response
        return $response;

    } // end::list



    /**
     * DATA AS OPTIONS
     * 
     * @return  array  $response
     */
    public function asOptions()
    {

        // Load the requested data
        $response = $this->select('id, name, code')
                        ->where([
                            $this->table . '.active'        => ACTIVE,
                            $this->table . '.deleted'       => INJECTION,
                            $this->table . '.deleted_by'    => null,
                        ])
                        ->get()
                        ->getResultArray();

        // Return the response
        return $response;

    } // end::asOptions

    

    /**
     * SET ROW
     * 
     * @param   int    $rowID
     * @param   array  $data
     * 
     * @return  array  $response
     */
    public function setRow(int $rowID = 0, array $data = []):?array
    {

        // Return the function response
        $code = 'danger';
        $message = lang('General.response.badRequest') . ' .';

        // Get all the required data
        $name = (! empty($data) && ! empty($data['name'])) ? $data['name'] : '';
        $permissionCode = (! empty($data) && ! empty($data['code'])) ? $data['code'] : '';

        // Process only if the required data are availables
        if(! empty($name) && ! empty($permissionCode)) {

            // Get the other data
            $rowID = (! empty($rowID) && intval($rowID) > 0) ? intval($rowID) : 0;
            $description = (! empty($data) && ! empty($data['description'])) ? $data['description'] : '';

            // Check if this is not duplicate data
            if(! $this->_isDuplicate([
                    $this->table . '.' . $this->primaryKey . ' !='  => $rowID,
                    $this->table . '.' . 'deleted'                  => INJECTION,
                    $this->table . '.' . 'deleted_at'               => null,
                    $this->table . '.' . 'deleted_by'               => null
                ], [
                    $this->table . '.name'  => $name,
                    $this->table . '.code'  => $permissionCode,
                ])
            ) {

                // Check wether we're facing an insertion or an update
                if(! empty($rowID))
                { // Update

                    // Try to get the row that correspond to the specified id
                    $existingRow = $this->list($rowID);

                    // Check wether the row exist or not
                    if(! empty($existingRow))
                    {

                        // Add the new destination
                        $updated = $this->update($rowID, [
                            'name'          => $name,
                            'code'          => $permissionCode,
                            'description'   => $description,
                        ]);

                        // Check if the row has been successfully created
                        if($updated)
                        {

                            // Set the response details
                            $code = 'success';
                            $message = lang('General.response.updated');

                        } // End if
                        else
                        {

                            // Set the response details
                            $code = 'warning';
                            $message = lang('General.response.server');

                        } // End else

                    } // End if
                    else
                    {

                        // Set the response details
                        $code = 'danger';
                        $message = lang('General.response.notFound');

                    } // End else

                } // End if
                else
                { // Creation

                    // Add the new row
                    $created = $this->save([
                        'name'          => $name,
                        'code'          => $permissionCode,
                        'description'   => $description,
                        'active'        => ACTIVE,
                    ]);

                    // Check if the row has been successfully created
                    if($created)
                    {

                        // Set the response details
                        $code = 'success';
                        $message = lang('General.response.created');

                    } // End if
                    else
                    {

                        // Set the response details
                        $code = 'warning';
                        $message = lang('General.response.server');

                    } // End else

                } // End else

            } // End if
            else
            {

                // Set the response details
                $code = 'danger';
                $message = lang('General.response.duplicated');

            } // End else

        } // End if

        // Set the response
        $response = [
            'code'      => $code,
            'message'   => ucfirst($message),
        ];

        // Return the response
        return $response;

    } // end::set



    /**
     * UDPATE ROW STATUS
     * ENABLE / DISABLE
     * 
     * @param   int  $rowID
     * @param   int  $status
     * 
     * @return  string $response
     */
    public function setStatus(int $rowID = 0, int $status = 0)
    {

        // Update the row status
        $response = $this->_status($rowID, $status);

        // Return the response
        return $response;

    } // end::setStatus



    /**
     * DELETE ROW
     * 
     * @param   int  $rowID
     * 
     * @return  string $response
     */
    public function deleteRow(int $rowID = 0)
    {

        // Update the row status
        $response = $this->_delete($rowID);

        // Return the response
        return $response;

    } // end::deleteRow



    /**
     * FORMAT ROW DATA
     * 
     * @param   array  $data
     * 
     * @return  array  $formatedData
     */
    public function _formatRowData($data = [], $isListPart = false, $validOnly = false)
    {

        // Set the default formatted data to return
        $formattedData = [];

        // Check wether the data to format is empty or not
        if(! empty($data))
        {

            // Get all the formatted data
            $id = (! empty($data['id']) && intval($data['id']) > 0) ? intval($data['id']) : 0;
            $name = (! empty($data['name'])) ? trim($data['name']) : '';
            $code = (! empty($data['code'])) ? trim($data['code']) : '';
            $description = (! empty($data['description'])) ? trim($data['description']) : '';
            $active = (! empty($data['active']) && intval($data['active']) > 0) ? intval($data['active']) : 0;
            $deleted = (! empty($data['deleted']) && intval($data['deleted']) > 0) ? intval($data['deleted']) : 0;
            $created_by = (! empty($data['created_by']) && intval($data['created_by']) > 0) ? intval($data['created_by']) : null;
            $deleted_by = (! empty($data['deleted_by']) && intval($data['deleted_by']) > 0) ? intval($data['deleted_by']) : null;
            $created_at = (! empty($data['created_at'])) ? trim($data['created_at']) : '';
            $updated_at = (! empty($data['updated_at'])) ? trim($data['updated_at']) : null;
            $deleted_at = (! empty($data['deleted_at'])) ? trim($data['deleted_at']) : null;

            // Check if only valid data are requested
            if($validOnly
                && (empty($id) || empty($name))
            )
            {

                // Exit the process
                return $formattedData;

            } // End if

            // Set the formatted data
            $formattedData = [
                'id'            => $id,
                'name'          => $name,
                'code'          => $code,
                'description'   => $description,
                'active'        => $active,
                'deleted'       => $deleted,
                'created_by'    => $created_by,
                'created_at'    => $created_at,
            ];

            // Check if data are required for a rows list
            if($isListPart)
            {

                // Set the formatted data to return
                return $formattedData;

            } // End if

            // Set the formatted data
            $formattedData['deleted_by']    = $deleted_by;
            $formattedData['updated_at']    = $updated_at;
            $formattedData['deleted_at']    = $deleted_at;

        } // End if

        return $formattedData;

    } // end::_formatRowData



    /**
     * FORMAT ROWS
     * 
     * @param   array  $data
     * 
     * @return  array  $formatedData
     */
    public function _formatRows($data = [], $isListPart = false, $validOnly = false)
    {

        // Set the default formatted data to return
        $formattedData = [];

        // Check wether the data to format is empty or not
        if(! empty($data))
        {

            // Loop through all the data then format each
            foreach($data as $row)
            {

                // Add the row the formatted data array
                $formattedData[] = $this->_formatRowData($row, $isListPart, $validOnly);


            } // End loop

        } // End if

        return $formattedData;

    } // end::_formatRows

} // end::PermissionsModel class